package com.hero.ui;

import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JPanel;

import com.hero.HeroDesigner;
import com.hero.Rules;
import com.hero.objects.GenericObject;
import com.hero.objects.enhancers.Enhancer;
import com.hero.objects.martialarts.Maneuver;
import com.hero.objects.powers.CompoundPower;
import com.hero.ui.dialog.AvailablePowersDialog;
import com.hero.ui.dialog.CompoundPowerDialog;
import com.hero.ui.widgets.PopupMessage;

/**
 * Copyright (c) 2000 - 2005, CompNet Design, Inc. All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, is prohibited unless the following conditions are met: 1.
 * Express written consent of CompNet Design, Inc. is obtained by the developer.
 * 2. Redistributions must retain this copyright notice. THIS SOFTWARE IS
 * PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * @author CompNet Design, Inc.
 * @version $Revision$
 */

public class CompoundPowerList extends GenericObjectList {

	private static final long serialVersionUID = 2447924922730948445L;

	protected JButton addPowerBtn;

	protected CompoundPower parent;

	public CompoundPowerList(CompoundPower power) {
		super("Compound Power", "Cost", "Power", "END", power.getPowers());
		parent = power;
		updateTotal();
	}

	/**
	 * Adds an object into the Compound Power/purchase list.
	 */
	@Override
	public void addObject(GenericObject o) {
		if (HeroDesigner.getActiveHero().isPrefab()) {
			super.addObject(o);
			return;
		}
		o.setPower(true);
		o.setMainPower(parent);
		boolean ret = true;
		boolean warned = false;
		Rules rules = HeroDesigner.getActiveHero().getRules();
		if (!warned && (rules.getAttackAPMaxResponse() > Rules.IGNORE)
				&& o.getTypes().contains("ATTACK")) {
			double active = o.getActiveCost();
			if (active > rules.getAttackAPMaxValue()) {
				PopupMessage popup = PopupMessage
						.getInstance(
								HeroDesigner.getAppFrame(),
								this,
								"This Attack exceeds the maximum Active Points for an Attack set in the campaign rules.",
								rules.getAttackAPMaxResponse() == Rules.WARN);
				popup.setVisible(true);
				warned = true;
				ret = rules.getAttackAPMaxResponse() == Rules.WARN;
			}
		}
		if (!warned && (rules.getDefenseAPMaxResponse() > Rules.IGNORE)
				&& o.getTypes().contains("DEFENSE")) {
			double active = o.getActiveCost();
			if (active > rules.getDefenseAPMaxValue()) {
				PopupMessage popup = PopupMessage
						.getInstance(
								HeroDesigner.getAppFrame(),
								this,
								"This Defense ability exceeds the maximum Active Points for a Defense ability set in the campaign rules.",
								rules.getDefenseAPMaxResponse() == Rules.WARN);
				popup.setVisible(true);
				warned = true;
				ret = rules.getDefenseAPMaxResponse() == Rules.WARN;
			}
		}
		if (ret) {
			super.addObject(o);
		}
	}

	/**
	 * Checks whether paste is allowed. Checks if the copy buffer is filled and,
	 * if so, if the item in the copy buffer may be pasted into a Compound
	 * Power.
	 */
	@Override
	protected boolean checkPaste() {
		GenericObject copy = HeroDesigner.getCopyBuffer();
		if (copy == null) {
			return false;
		} else {
			if (copy instanceof Maneuver) {
				return false;
			} else if (copy instanceof Enhancer) {
				return false;
			} else if (copy instanceof CompoundPower) {
				return false;
			} else if (copy instanceof com.hero.objects.List) {
				com.hero.objects.List list = (com.hero.objects.List) copy;
				for (int i = 0; i < list.getObjects().size(); i++) {
					if (list.getObjects().get(i) instanceof Maneuver) {
						return false;
					}
				}
				return true;
			} else {
				return true;
			}
		}
	}

	@Override
	protected boolean checkReplace(GenericObject replacement,
			GenericObject replacee) {
		if (HeroDesigner.getActiveHero().isPrefab()) {
			return true;
		}
		boolean ret = true;
		boolean warned = false;
		Rules rules = HeroDesigner.getActiveHero().getRules();
		if (!warned && (rules.getAttackAPMaxResponse() > Rules.IGNORE)
				&& replacement.getTypes().contains("ATTACK")) {
			double active = replacement.getActiveCost();
			if ((active > rules.getAttackAPMaxValue())
					&& (active > replacee.getActiveCost())) {
				PopupMessage popup = PopupMessage
						.getInstance(
								HeroDesigner.getAppFrame(),
								this,
								"This Attack exceeds the maximum Active Points for an Attack set in the campaign rules.",
								rules.getAttackAPMaxResponse() == Rules.WARN);
				popup.setVisible(true);
				warned = true;
				ret = rules.getAttackAPMaxResponse() == Rules.WARN;
			}
		}
		if (!warned && (rules.getDefenseAPMaxResponse() > Rules.IGNORE)
				&& replacement.getTypes().contains("DEFENSE")) {
			double active = replacement.getActiveCost();
			if ((active > rules.getDefenseAPMaxValue())
					&& (active > replacee.getActiveCost())) {
				PopupMessage popup = PopupMessage
						.getInstance(
								HeroDesigner.getAppFrame(),
								this,
								"This Defense ability exceeds the maximum Active Points for a Defense ability set in the campaign rules.",
								rules.getDefenseAPMaxResponse() == Rules.WARN);
				popup.setVisible(true);
				warned = true;
				ret = rules.getDefenseAPMaxResponse() == Rules.WARN;
			}
		}
		return ret;
	}

	@Override
	protected JPanel getButtonPanel() {
		JPanel buttons = new JPanel(new FlowLayout(FlowLayout.CENTER));
		buttons.add(downBtn);
		buttons.add(addPowerBtn);
		buttons.add(editBtn);
		buttons.add(insertBtn);
		buttons.add(deleteBtn);
		buttons.add(upBtn);
		buttons.setOpaque(false);
		return buttons;
	}

	@Override
	protected void initListeners() {
		super.initListeners();
		addPowerBtn.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				Container con = getParent();
				while ((con != null) && !(con instanceof CompoundPowerDialog)) {
					con = con.getParent();
				}
				AvailablePowersDialog dialog = new AvailablePowersDialog(
						parent, (JDialog) con);
				dialog.setLocationRelativeTo(CompoundPowerList.this);
				dialog.setVisible(true);
				updateTotal();
				((CompoundPowerDialog) con).updateValues();
			}
		});
	}

	@Override
	protected void initWidgets() {
		super.initWidgets();
		addPowerBtn = new JButton("Add");
	}

	/**
	 * Do nothing method...just sets the title.
	 */
	@Override
	public void setTotalCost() {
		setTitle("Component Powers");
	}
}